package ro.sync.lexer.java;
import ro.sync.lexer.AbstractLexer;

@SuppressWarnings("unused")
%%

%public 
%class JavaLexer
%extends AbstractLexer
%unicode
%char
%type ro.sync.lexer.Symbol
 
%scanerror ro.sync.lexer.LexerException

%{
    /**
     * Text
     */
    private static final byte SYM_TEXT              = JAVATokens.TEXT;
    
    /**
     * Java keywords
     */
    private static final byte SYM_KEYWORD           = JAVATokens.KEYWORD;
    
    /**
     * Java identifiers 
     */
    private static final byte SYM_IDENTIFIER        = JAVATokens.IDENTIFIER;
    
    /**
     * Labels
     */
    private static final byte SYM_LABEL             = JAVATokens.LABEL;
    
    /**
     * Java operators
     */
    private static final byte SYM_OPERATOR          = JAVATokens.OPERATOR;
    
    /**
     * Class member delimiter
     */
    private static final byte SYM_DOT               = JAVATokens.DOT;
    
    /**
     * Brackets
     */
    private static final byte SYM_BRACKET           = JAVATokens.BRACKET;
    private static final byte SYM_CURLY_BRACKET     = JAVATokens.CURLY_BRACKET;
    private static final byte SYM_SQUARE_BRACKET    = JAVATokens.SQUARE_BRACKET;
    
    /**
     * Comments
     */
    private static final byte SYM_COMMENT           = JAVATokens.COMMENT;
    
    /**
     * Java doc
     */
    private static final byte SYM_JAVA_DOC          = JAVATokens.JAVA_DOC;
    
    /**
     * String literal
     */
    private static final byte SYM_STRING            = JAVATokens.STRING;
    
    /**
     * Char literal
     */
    private static final byte SYM_CHAR              = JAVATokens.CHAR;

    /**
     * Numbers
     */
    private static final byte SYM_NUMBER            = JAVATokens.NUMBER;
    
    /**
     * Comma
     */
    private static final byte SYM_COMMA             = JAVATokens.COMMA;
    
    /**
     * JSP markup.
     */
    private static final byte SYM_JSP_MARKUP        = JAVATokens.JSP_MARKUP;
    
    /**
     * Create an empty lexer, yyreset will be called later to reset and assign
     * the reader
     */
    public JavaLexer() {
        super();
    }
    
    public String getName() {
      return JAVA_LEXER;
    }
%}

%xstate COMMENT, JAVA_DOC, STRING, CHAR_LITERAL

Operator = ";" |  "=" | ">" | "<" | "!" | "~" | "?" | ":" | "==" | "<=" | ">=" | "!=" 
         | "&&" | "||" | "++" | "--" | "+" | "-" | "*" | "/" | "&" | "|" | "^" | "%" | "<<" | ">>"
         | ">>>" | "+=" | "-=" | "*=" | "/=" | "&=" | "|=" | "^=" | "%=" | "<<=" | ">>=" | ">>>="

Keyword = "abstract" | "boolean" | "break" | "byte" | "case" | "catch" | "char" | "class" | "const"
        | "continue" | "do" | "double" | "enum" | "else" | "extends" | "final" | "finally" | "float"
        | "for" | "default" | "implements" | "import" | "instanceof" | "int" | "interface" | "long" 
        | "native" | "new" | "goto" | "if" | "public" | "short" | "super" | "switch" | "synchronized"
        | "package" | "private" | "protected" | "transient" | "return" | "void" | "static" | "while"
        | "this" | "throw" | "throws" | "try" | "volatile" | "strictfp" | "true" | "false" | "null" 

Identifier = [a-zA-Z_][a-zA-Z0-9_]*
Label = {Identifier} ":"

// Numbers
Digit = [0-9]
Integer = {Digit}+
Long = {Integer} [lL]

HexDigit          = [0-9a-fA-F]
HexInteger = 0 [xX] {HexDigit}+
HexLong = {HexInteger} [lL]

/* floating point literals */
F1 = {Digit}+ \. {Digit}* 
F2 = \. {Digit}+ 
F3 = {Digit}+ 
Exponent = [eE] [+-]? {Digit}+
        
Double = ({F1}|{F2}|{F3}) {Exponent}?
Float  = {Double} [fF]

Number = {Integer} | {Long} | {HexInteger} | {HexLong} | {Double} | {Float}

// All characters
Char = .
GeneralChar = [^ \t\-\[\]\^=!;.+|<>()*%{}/?:,\"\']

%%

<YYINITIAL> 
{
    "<%" | "%>"                 {   return symbol(SYM_JSP_MARKUP);      }
    ","                         {   return symbol(SYM_COMMA);           }
    {Keyword}                   {   return symbol(SYM_KEYWORD);         }
    {Identifier}                {   return symbol(SYM_IDENTIFIER);      }
    {Operator}                  {   return symbol(SYM_OPERATOR);        }
    {Label}                     {   return symbol(SYM_LABEL);           }
    "."                         {   return symbol(SYM_DOT);             }
    
    "\""                        {   
                                    cLen++;
                                    yybegin(STRING);          
                                }
    
    "\'"                        {
                                    cLen++;
                                    yybegin(CHAR_LITERAL);            
                                }
    
    {Number}                    {   return symbol(SYM_NUMBER);          }
    
    "/*"                        { 
                                    yypushback(2);
                                    yybegin(COMMENT);   
                                }
    "/**"                       { 
                                    yypushback(3);
                                    yybegin(JAVA_DOC);   
                                }
    "//" {Char}*                {   return symbol(SYM_COMMENT);         }
    "(" | ")"                   {   return symbol(SYM_BRACKET);         }
    "[" | "]"                   {   return symbol(SYM_SQUARE_BRACKET);  }
    "{" | "}"                   {   return symbol(SYM_CURLY_BRACKET);   }
    
    [ \t]+                      {   return symbol(SYM_TEXT);            }
    {GeneralChar}+              {   return symbol(SYM_TEXT);            }
}

<STRING>
{
    "\\\\" | "\\\""             {   cLen += 2;                          }
    [^\"]                       {   cLen++;                             }
    "\""                        {
                                    cLen++;
                                    yybegin(YYINITIAL);
                                    return flush(SYM_STRING);
                                }
    <<EOF>>                     {   
                                    yybegin(YYINITIAL);
                                    return flush(SYM_STRING);
                                }
}

<CHAR_LITERAL>{
    "\\\\" | "\\\'"             {   cLen += 2;                          }
    ("u" [0-9a-fA-f]+)          {   cLen += yylength();                 }
    [^\']                       {   cLen++;                             }
    "\'"                        {
                                    cLen++;
                                    yybegin(YYINITIAL);
                                    return flush(SYM_CHAR);
                                }
    <<EOF>>                     {   
                                    yybegin(YYINITIAL);
                                    return flush(SYM_CHAR);
                                }
}

<COMMENT> 
{
    "*/"                        {
                                    cLen += 2;
                                    yybegin(YYINITIAL);
                                    return flush(SYM_COMMENT);
                                }
    {Char}                      {   cLen++;                             }
    <<EOF>>                     {   
                                    return flush(SYM_COMMENT);
                                }
}

<JAVA_DOC> 
{
    "*/"                        {
                                    cLen += 2;
                                    yybegin(YYINITIAL);
                                    return flush(SYM_JAVA_DOC);
                                }
    {Char}                      {   cLen++;                             }
    <<EOF>>                     {   
                                    return flush(SYM_JAVA_DOC);
                                }
}