// -*- Mode: vala; indent-tabs-mode: nil; tab-width: 4 -*-
/*-
 * Copyright (c) 2011-2015 Maya Developers (http://launchpad.net/maya)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Maxwell Barvian
 *              Corentin Noël <corentin@elementaryos.org>
 */

namespace Maya.View {

/**
 * Represents the entire date grid as a table.
 */
public class Grid : Gtk.Grid {

    Gee.HashMap<uint, GridDay> data;

    public Calendar.Util.DateRange grid_range { get; private set; }

    /*
     * Event emitted when the day is double clicked or the ENTER key is pressed.
     */
    public signal void on_event_add (DateTime date);

    public signal void selection_changed (DateTime new_date);
    private GridDay selected_gridday;

    private static Gtk.CssProvider style_provider;

    private GridDay today_widget = null;

    static construct {
        style_provider = new Gtk.CssProvider ();
        style_provider.load_from_resource ("/io/elementary/calendar/Grid.css");
    }

    construct {
        // Gtk.Grid properties
        insert_column (7);
        set_column_homogeneous (true);
        set_row_homogeneous (true);
        column_spacing = 0;
        row_spacing = 0;

        data = new Gee.HashMap<uint, GridDay> ();
        events |= Gdk.EventMask.SCROLL_MASK;
        events |= Gdk.EventMask.SMOOTH_SCROLL_MASK;

        unowned var time_manager = Calendar.TimeManager.get_default ();
        time_manager.on_update_today.connect (callback_update_today);
    }

    /** Update the names of the current and previous "Today" cells, if necessary.
     */
    void callback_update_today () {
        var old_today_widget = today_widget;

        // Add label to the new widget, if it exists and is not up to date
        var today = Calendar.Util.datetime_strip_time (new DateTime.now_local ());
        var today_hash = day_hash (today);
        if (data.has_key (today_hash)) { // Today cell is on the grid
            var new_today_widget = data.get (today_hash);

            if (new_today_widget.name == "today") {
                debug ("Today cell already up to date. Nothing to do.");
                return;
            } else {
                new_today_widget.name = "today";
                today_widget = new_today_widget;
            }
        } else {
            debug ("Today out of range of calendar grid.");
            if (today_widget != null) {
                today_widget.name = "MayaViewGridDay";
            }
            today_widget = null;
            return;
        }

        // Remove label from old widget, if necessary.
        if (old_today_widget != null) {
            old_today_widget.name = "MayaViewGridDay";
        } else {
            debug ("No previous today widget to update. Nothing to do.");
        }
    }

    void on_day_focus_in (GridDay day) {
        if (selected_gridday != null)
            selected_gridday.set_selected (false);
        var selected_date = day.date;
        selected_gridday = day;
        day.set_selected (true);
        day.set_state_flags (Gtk.StateFlags.FOCUSED, false);
        selection_changed (selected_date);

        Maya.Application.saved_state.set_string ("selected-day", selected_date.format ("%Y-%j"));

        var calmodel = Calendar.EventStore.get_default ();
        var date_month = selected_date.get_month () - calmodel.month_start.get_month ();
        var date_year = selected_date.get_year () - calmodel.month_start.get_year ();
        if (date_month != 0 || date_year != 0) {
            calmodel.change_month (date_month);
            calmodel.change_year (date_year);
        }
    }

    public void focus_date (DateTime date) {
        debug (@"Setting focus to @ $(date)");
        var date_hash = day_hash (date);
        if (data.has_key (date_hash) == true) {
            var day_widget = data.get (date_hash);
            day_widget.grab_focus ();
            on_day_focus_in (day_widget);
        }
    }

    /**
     * Sets the given range to be displayed in the grid. Note that the number of days
     * must remain the same.
     */
    public void set_range (Calendar.Util.DateRange new_range, DateTime month_start) {
        var today = new DateTime.now_local ();

        Gee.List<DateTime> old_dates;
        if (grid_range == null) {
            old_dates = new Gee.ArrayList<DateTime> ();
        } else {
            old_dates = grid_range.to_list ();
        }

        var new_dates = new_range.to_list ();

        var data_new = new Gee.HashMap<uint, GridDay> ();

        // Assert that a valid number of weeks should be displayed
        assert (new_dates.size % 7 == 0);

        // Create new widgets for the new range

        int i=0;
        int col = 0, row = 0;

        for (i = 0; i < new_dates.size; i++) {
            var new_date = new_dates [i];
            GridDay day;
            if (i < old_dates.size) {
                // A widget already exists for this date, just change it

                var old_date = old_dates [i];
                day = update_day (data[day_hash (old_date)], new_date, today, month_start);

            } else {
                // Still update_day to get the color of etc. right
                day = update_day (new GridDay (new_date), new_date, today, month_start);
                day.on_event_add.connect ((date) => on_event_add (date));
                day.scroll_event.connect ((event) => {scroll_event (event); return false;});
                day.focus_in_event.connect ((event) => {
                    on_day_focus_in (day);
                    return false;
                });

                if (col == 0) {
                    unowned Gtk.StyleContext day_context = day.get_style_context ();
                    day_context.add_provider (style_provider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION);
                    day_context.add_class ("firstcol");
                }

                attach (day, col, row);
                day.show_all ();
            }

            col = (col + 1) % 7;
            row = (col == 0) ? row + 1 : row;
            data_new.set (day_hash (new_date), day);
        }

        // Destroy the widgets that are no longer used
        while (i < old_dates.size) {
            // There are widgets remaining that are no longer used, destroy them
            var old_date = old_dates [i];
            var old_day = data.get (day_hash (old_date));

            old_day.destroy ();
            i++;
        }

        data.clear ();
        data.set_all (data_new);

        grid_range = new_range;
    }

    /**
     * Updates the given GridDay so that it shows the given date. Changes to its style etc.
     */
    GridDay update_day (GridDay day, DateTime new_date, DateTime today, DateTime month_start) {
        if (new_date.get_day_of_year () == today.get_day_of_year () && new_date.get_year () == today.get_year ()) {
            day.name = "today";
            today_widget = day;
        }

        day.in_current_month = new_date.get_month () == month_start.get_month ();

        day.date = new_date;
        return day;
    }

    /**
     * Puts the given event on the grid.
     */
    public void add_event (ECal.Component event) {
        foreach (var grid_day in data.values) {
            if (Calendar.Util.ecalcomponent_is_on_day (event, grid_day.date)) {
                var button = new EventButton (event);
                grid_day.add_event_button (button);
            }
        }
    }

    uint day_hash (DateTime date) {
        return date.get_year () * 10000 + date.get_month () * 100 + date.get_day_of_month ();
    }

    /**
     * Removes the given event from the grid.
     */
    public void remove_event (ECal.Component event) {
        foreach (var grid_day in data.values) {
            grid_day.remove_event (event);
        }
    }

    public void update_event (ECal.Component event) {
        foreach (var grid_day in data.values) {
            if (Calendar.Util.ecalcomponent_is_on_day (event, grid_day.date)) {
                if (!grid_day.update_event (event)) {
                    var button = new EventButton (event);
                    grid_day.add_event_button (button);
                }
            } else {
                grid_day.remove_event (event);
            }
        }
    }

    /**
     * Removes all events from the grid.
     */
    public void remove_all_events () {
        foreach (var grid_day in data.values) {
            grid_day.clear_events ();
        }
    }
}
}
