#!/bin/bash
#
#### generate_po_files.sh   Written by **peter1000** (for gettext v.0.19.7) contributed to NuTyX-Linux 
#
#   
#  Copyright (c) 2016 by NuTyX Team (http://nutyx.org)
#  Copyright (c) 2016 **peter1000** (https://github.com/peter1000)
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
#  USA.
#
# Read the COPYING file for the complete license.

#**********************************************************************************************************************
#
#   Generate and update `.po` files: **peter1000**
#
#   1. This will extract an original language (as written in the source file) .pot file
#
#   2. Generate for all defined locale individual .po files if they exit they will be merged with the differences
#
#       1. If the locale 'po' file exists: only new messages will be merged: a backup of the previous will be made
#
#       2. If the locale 'po' file does notexists:
#           In this case an original po files is generated: with an `_empty` added to the filename
#
#           * The name should be manually changed to the same basename as the original po file(removing `_empty`) 
#           
#           * the header tags should be updated and the messages should be translated
#
#
#**********************************************************************************************************************
#
#                                     >>>     Filling in the: Header Entry     <<<                                     
#
#    Check all but pay special attention to: `Last-Translator`, `Language-Team` `Language`
#
#   Example:
#
#       "Last-Translator: Thierry Nuttens <tnut@example.org>\n"
#       "Language-Team: French <tnut@example.org>\n"
#       "Language: fr_FR\n"
#
#   for more info see: <https://www.gnu.org/software/gettext/manual/html_node/Header-Entry.html>
#
#**********************************************************************************************************************

THIS_SCRIPT_PATH="$(readlink -f "$(type -P $0 || echo $0)")"
THIS_SCRIPT_DIR="$(dirname "$THIS_SCRIPT_PATH")"

#**********************************************************************************************************************
#
#   DEFAULT CONFIGURATION
#

# NOTE: UTF-8 language/locale are expected
LANGUAGES=(
    'de_DE.UTF-8'
    'en_US.UTF-8' 
    'fr_FR.UTF-8'  
    'pt_BR.UTF-8'
)
 
PACKAGE_NAME='cards'

SRCFILE_PATH="$THIS_SCRIPT_DIR/pkgmk.in"

PO_DIR="$THIS_SCRIPT_DIR/po"
COPYRIGHT_START_YEAR="2016"
COPYRIGHT_HOLDER="NuTyX Team"
PROJECT_URL="https://github.com/NuTyX/cards"

#
#   END CONFIGURE
#
#**********************************************************************************************************************

ALL_OFF="\e[0m"
BOLD="\e[1m"
GREEN="${BOLD}\e[32m"

info1() {
	local mesg=$1; shift
	printf "${GREEN}==>${ALL_OFF}${BOLD} ${mesg}${ALL_OFF}\n" "$@" >&1
}

info2() {
	local mesg=$1; shift
	printf "${BOLD}    ${mesg}${ALL_OFF}\n" "$@" >&1
}

get_basename() {
    printf "%s\n" "${1##*/}"
}

# Generate original .pot file from source: `SRCFILE_PATH`
# Writes the output to PO_DIR/FILE_NAME.pot
generate_pot_file() {
    local POT_FILE_PATH="$1"
    local SRCFILE_BASENAME="$2"
    local SRCFILE_DIR=="$(dirname "$SRCFILE_DIR")"
    local CURRENT_YEAR="$(date +%Y)"
    
    # remove any existing 
    rm -f $POT_FILE_PATH
    
    info1 "$(gettext "Generating original 'pot' file...SOURCE: <%s>")" "$SRCFILE_BASENAME"

	pushd "$SRCFILE_DIR" &>/dev/null
    xgettext \
        --output "$POT_FILE_PATH" \
        --language=Shell \
        --package-name="$PACKAGE_NAME" \
        --copyright-holder="$COPYRIGHT_HOLDER" \
        --msgid-bugs-address="$PROJECT_URL/issues" \
        --from-code="UTF-8" \
        --force-po  \
        --no-wrap   \
        "$SRCFILE_BASENAME"
     
    # update the pot header
    sed -i.bak "
            s|SOME DESCRIPTIVE TITLE.|Package: <$PACKAGE_NAME> Source File: <$SRCFILE_BASENAME>: original .pot file, autogenerated by <$(get_basename "$THIS_SCRIPT_PATH")>.|;
            s|Copyright (C) YEAR|Copyright (C) $COPYRIGHT_START_YEAR - $CURRENT_YEAR|;
            s|FIRST AUTHOR <EMAIL@ADDRESS>, YEAR.|$COPYRIGHT_HOLDER <$PROJECT_URL>, $COPYRIGHT_START_YEAR.|;
            s|LL@li.org|$PROJECT_URL|;
            s|charset=CHARSET|charset=UTF-8|;
    " "$POT_FILE_PATH"
    
    rm -f "$POT_FILE_PATH.bak"
    
    info2 "$(gettext "  >>> PATH: <%s>")" "$POT_FILE_PATH"
    
    popd &>/dev/null
}

main() {
    local EMPTY_PO_PATH FINAL_PO_PATH LOCALE_PO_DIR
    local SRCFILE_BASENAME="$(get_basename "$SRCFILE_PATH")"
    local SRCFILE_BASENAME_NO_EXT="${SRCFILE_BASENAME%.*}"                   
    local POT_FILE_PATH="$PO_DIR/${SRCFILE_BASENAME_NO_EXT}.pot"

    mkdir -p "$PO_DIR"

    generate_pot_file "$POT_FILE_PATH" "$SRCFILE_BASENAME" 
    
    info1 "$(gettext "Generating all locale 'po' files...")"
    
    for locale in "${LANGUAGES[@]}"; do
        LOCALE_PO_DIR="$PO_DIR/$locale/LC_MESSAGES"
        mkdir -p "${LOCALE_PO_DIR}"

        info1 "$(gettext "Proccessing locale <%s>...")" "$locale"
        
        EMPTY_PO_PATH="$LOCALE_PO_DIR/${SRCFILE_BASENAME_NO_EXT}_empty.po"
        FINAL_PO_PATH="$LOCALE_PO_DIR/${SRCFILE_BASENAME_NO_EXT}.po"

        if [[ -f "$FINAL_PO_PATH" ]]; then
            msgmerge \
                --update \
                --backup=simple \
                --no-wrap \
                "$FINAL_PO_PATH" "$POT_FILE_PATH"
        else
            rm -f "$EMPTY_PO_PATH"

            msginit \
                --no-wrap \
                --no-translator \
                --locale="$locale" \
                --input="$POT_FILE_PATH" \
                --output-file="$EMPTY_PO_PATH"
                
            # update the po header: with our available data: should be manually set
            sed -i.bak "
                    s|Last-Translator: Automatically generated|Last-Translator: $COPYRIGHT_HOLDER <$PROJECT_URL>|;
                    s|Language-Team: none|Language-Team: LANGUAGE <$PROJECT_URL>|;
            " "$EMPTY_PO_PATH"
            
            rm -f "$EMPTY_PO_PATH.bak"
        fi
    done
}


#**********************************************************************************************************************

main
